//*************************************************************************************************
//
//	Description:
//		Unlit - temp shader for simple single diffuse texture material with no lighting
//
//	<P> Copyright (c) 2006 Blimey! Games Ltd. All rights reserved.
//
//	Author: 
//		Matt Hobbs
//
//	History:
//
//	<TABLE>
//		\Author         Date        Version       Description
//		--------        -----       --------      ------------
//		MHobbs			    11/01/2006  0.1						Created
//		TNettleship			11/20/2006  0.2						Added translucent techniques
//		TMann						11/24/2006	1.2						Added _WIN_GL_ support
//		TMann						11/27/2006	1.3						Texture filtering for GL
//		TMann						12/11/2006	1.4						PS3 version
//		TNettleship			04/19/2007	2.0						Added shader specialisation.
//		TNettleship     07/24/2007  2.01          Made sure samplers aren't using anisotropic filtering.
//		TNettleship     08/17/2007  2.02					Removed texture bias.
//		TNettleship			10/23/2007	2.03					Converted to on-load rendermode behaviour binding.
//	<TABLE>
//
//*************************************************************************************************

#include "stddefs.fxh"
#include "specialisation_globals.fxh"



//-----------------------------------------------------------------------
//
// Input parameters
//




//
// Transforms
//
float4x4 worldviewproj : WorldViewProjection
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
	bool dynamic = true;
>;



//
// Channel mappings (max only)
//

//
// N.B. Max contains a bug which means the colour channel must NOT be mapped to texcoord0.
// The first UV coord channel MUST be mapped to texcoord0 or the basis vectors for normal
// mapping will be screwed up. (e.g. there's some bit of code deep within max which assumes
// this setup when calculating the basis vectors)
//

#ifdef _3DSMAX_

// First UV channel
int texcoord0 : Texcoord
<
	string UIWidget = "None";
	int Texcoord = 0;
	int MapChannel = 1;
	int RuntimeTexcoord = 0;
	bool export = false;
> = 0;

// Vertex colour channel
int texcoord1 : Texcoord
<
	string UIWidget = "None";
	int Texcoord = 1;
	int MapChannel = 0;
	bool ColorChannel = true;
	bool export = false;
> = 0;

#endif


//
// Textures
//

#ifdef _3DSMAX_
texture diffuseTexture : DiffuseMap											// Diffuse colour in RGB, translucency in alpha
#else
texture diffuseTexture : TEXTURE											// Diffuse colour in RGB, translucency in alpha
#endif
<
	string UIName = "Diffuse Texture";
	bool appEdit = true;
>;

float4 staticColour : Color												// static colour multiplier
<
	string UIName = "Static Colour";
	string UIType = "ColorSwatch";
	bool appEdit = true;
> = {1.0f, 1.0f, 1.0f, 1.0f};




//-----------------------------------------------------------------------
//
// Samplers
//

sampler2D diffuseMap : SAMPLER 
< 
	SET_SRGB_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="diffuseTexture"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < diffuseTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_SRGB_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};

//-----------------------------------------------------------------------
//
// Functions
//




//-----------------------------------------------------------------------
//
// Vertex Shader(s)
//

// Input structure
struct VSINPUT
{
	float3 position : POSITION;														// Object space position
#ifdef _3DSMAX_
	float4 colour   : TEXCOORD1;													// Vertex colour
	float2 texCoord : TEXCOORD0;													// UV channel 1 texture coord - N.B. MAx requires that texcoord0 is a geometric channel
																												// as it implicitly uses that to calculate the tangent space coordinate frame.
#else
	float4 colour   : COLOR0;															// Vertex colour
	float2 texCoord : TEXCOORD0;													// UV channel 1 texture coord
#endif
};


// Output structure
struct VSOUTPUT
{
	float4 position	: POSITION;														// View-coords position
	float4 colour		: COLOR0;															// Vertex colour
	float2 texCoord	: TEXCOORD0;													// UV coords for texture channel 0
};



//-----------------------------------------------------------------------
//
// Vertex shader code
//

VSOUTPUT UnlitVertexShader( VSINPUT _input )
{
	VSOUTPUT _output;

	// Copy simple invariant params to output structure
	_output.colour	 = _input.colour;
	_output.texCoord = _input.texCoord;

	// Calculate clip-space position of the vertex
	_output.position = mul( float4( _input.position, 1.0f ), worldviewproj );

	return _output;
}



//-----------------------------------------------------------------------
//
// Fragment Shader(s)
//

// Input structure
struct PSINPUT
{
	float4 colour		: COLOR0;													// Vertex colour
	float2 texCoord		: TEXCOORD0;										// UV coords for texture channel 0
};


// Output structure
struct PSOUTPUT
{
	COLOUR_OUTPUT_TYPE Colour : COLOR0;
};



//-----------------------------------------------------------------------
//
// Fragment shader code
//

PSOUTPUT UnlitFragmentShader( PSINPUT _input )
{
	PSOUTPUT _output;

	// Read textures
	float4 diffuseTexColour = tex2D( diffuseMap, _input.texCoord );
	
	// Calculate pixel colour
	float4 accumulatedColour = staticColour * diffuseTexColour * _input.colour;

	_output.Colour = CalculateOutputPixel(accumulatedColour);

	return _output;
}


PSOUTPUT UnlitHDRFragmentShader( PSINPUT _input )
{
	PSOUTPUT _output;

	// Read texture (rgbdiv8 format)
	float4 rgbDiv8TexColour = tex2D( diffuseMap, _input.texCoord );

	// Calculate RGB pixel colour from RGBdiv8 source
	float4 hdrColour = float4( rgbDiv8TexColour.rgb / rgbDiv8TexColour.a, 1.0f );

	// Calculate alpha value (equal to clamped luminosity)
	hdrColour.a = saturate(  ( hdrColour.r + hdrColour.g + hdrColour.b ) * 0.333f );

	// Calculate pixel colour
	float4 accumulatedColour = staticColour * hdrColour * _input.colour;

	_output.Colour = CalculateOutputPixel(accumulatedColour);

	return _output;
}


PSOUTPUT UnlitLowDetailFragmentShader( PSINPUT _input )
{
	PSOUTPUT _output;

	// Read textures
	float4 diffuseTexColour = tex2D( diffuseMap, _input.texCoord );
	
	// Calculate pixel colour
	float4 accumulatedColour = staticColour * diffuseTexColour * _input.colour;

	_output.Colour = CalculateLowDetailOutputPixel(accumulatedColour);

	return _output;
}



//-----------------------------------------------------------------------
//
// Technique(s)
//

technique Unlit
<
	bool supportsSpecialisedLighting = false;
	bool preservesGlobalState = true;
	string normalBehaviour		= "ERMB_RENDER";
	string normalTechnique		= "Unlit";
	int    normalDeferredID		= 0;
	string zprimeBehaviour		= "ERMB_DONT_RENDER";
	string zprimeDOFBehaviour	= "ERMB_DONT_RENDER";
	string shadowGenBehaviour = "ERMB_RENDER_DEFAULT";
	string lowDetailBehaviour	= "ERMB_RENDER";
	string lowDetailTechnique	= "_Unlit_LowDetail";
	int    lowDetailDeferredID = 0;
>
{
	pass Pass0
#ifdef _3DSMAX_
	<
		bool ZEnable = true;
		bool ZWriteEnable = true;
		bool AlphaBlendEnable = false;
	>
#endif
	{
#ifdef _3DSMAX_
		ZEnable = true;
		ZWriteEnable = true;
		AlphaBlendEnable = false;
#endif

#if defined (_PS3_)		
		VertexShader = compile sce_vp_rsx UnlitVertexShader();
		PixelShader = compile sce_fp_rsx UnlitFragmentShader();
#else		
		VertexShader = compile vs_3_0 UnlitVertexShader();
		PixelShader = compile ps_3_0 UnlitFragmentShader();
#endif		
	}
}



technique Unlit_Trans
<
	bool preservesGlobalState = true;
	string normalBehaviour		= "ERMB_RENDER";
	string normalTechnique		= "Unlit";
	int    normalDeferredID		= 2;
	string zprimeBehaviour		= "ERMB_DONT_RENDER";
	string zprimeDOFBehaviour	= "ERMB_DONT_RENDER";
	string shadowGenBehaviour = "ERMB_DONT_RENDER";
	string lowDetailBehaviour	= "ERMB_RENDER";
	string lowDetailTechnique	= "_Unlit_LowDetail";
	int    lowDetailDeferredID = 0;
>
{
	pass Pass0
#ifdef _3DSMAX_
	<
		bool ZEnable = true;
		bool ZWriteEnable = false;
		bool AlphaBlendEnable = true;
		string DestBlend = "INVSRCALPHA";
		string SrcBlend = "SRCALPHA";
		string BlendOp = "ADD";
	>
#endif
	{
#ifdef _3DSMAX_
		ZEnable = true;
		ZWriteEnable = false;
		AlphaBlendEnable = true;
		DestBlend = INVSRCALPHA;
		SrcBlend = SRCALPHA;
		BlendOp = ADD;
#endif

#ifdef _3DSMAX_
		// The rendermode mapping table above maps rendering of this technique
		// to other techniques in all modes, so it doesn't need its own compiled
		// shaders, except in max.
		VertexShader = compile vs_3_0 UnlitVertexShader();
		PixelShader = compile ps_3_0 UnlitFragmentShader();
#endif
	}
}



technique Unlit_Trans_NoDefer
<
	bool preservesGlobalState = true;
	string normalBehaviour		= "ERMB_RENDER";
	string normalTechnique		= "Unlit";
	int    normalDeferredID		= 0;
	string zprimeBehaviour		= "ERMB_DONT_RENDER";
	string zprimeDOFBehaviour	= "ERMB_DONT_RENDER";
	string shadowGenBehaviour = "ERMB_DONT_RENDER";
	string lowDetailBehaviour	= "ERMB_RENDER";
	string lowDetailTechnique	= "_Unlit_LowDetail";
	int    lowDetailDeferredID = 0;
>
{
	pass Pass0
#ifdef _3DSMAX_
	<
		bool ZEnable = true;
		bool ZWriteEnable = false;
		bool AlphaBlendEnable = true;
		string DestBlend = "INVSRCALPHA";
		string SrcBlend = "SRCALPHA";
		string BlendOp = "ADD";
	>
#endif
	{
#ifdef _3DSMAX_
		ZEnable = true;
		ZWriteEnable = false;
		AlphaBlendEnable = true;
		DestBlend = INVSRCALPHA;
		SrcBlend = SRCALPHA;
		BlendOp = ADD;
#endif

#ifdef _3DSMAX_
		// The rendermode mapping table above maps rendering of this technique
		// to other techniques in all modes, so it doesn't need its own compiled
		// shaders, except in max.
		VertexShader = compile vs_3_0 UnlitVertexShader();
		PixelShader = compile ps_3_0 UnlitFragmentShader();
#endif
	}
}


technique Unlit_Trans_NoDefer_Additive
<
	bool preservesGlobalState = true;
	string normalBehaviour		= "ERMB_RENDER";
	string normalTechnique		= "Unlit";
	int    normalDeferredID		= 2;
	string zprimeBehaviour		= "ERMB_DONT_RENDER";
	string zprimeDOFBehaviour	= "ERMB_DONT_RENDER";
	string shadowGenBehaviour = "ERMB_DONT_RENDER";
	string lowDetailBehaviour	= "ERMB_DONT_RENDER";
>
{
	pass Pass0
#ifdef _3DSMAX_
	<
		bool ZEnable = true;
		bool ZWriteEnable = false;
		bool AlphaBlendEnable = true;
		string DestBlend = "ONE";
		string SrcBlend = "SRCALPHA";
		string BlendOp = "ADD";
	>
#endif
	{
#ifdef _3DSMAX_
		ZEnable = true;
		ZWriteEnable = FALSE;
		AlphaBlendEnable = true;
		DestBlend = ONE;
		SrcBlend = SRCALPHA;
		BlendOp = ADD;
#endif

#ifdef _3DSMAX_
		// The rendermode mapping table above maps rendering of this technique
		// to other techniques in all modes, so it doesn't need its own compiled
		// shaders, except in max.
		VertexShader = compile vs_3_0 UnlitVertexShader();
		PixelShader = compile ps_3_0 UnlitFragmentShader();
#endif
	}
}


technique Unlit_Trans_HDR
<
	bool preservesGlobalState = true;
	string normalBehaviour		= "ERMB_RENDER";
	string normalTechnique		= "Unlit_Trans_HDR";
	int    normalDeferredID		= 2;
	string zprimeBehaviour		= "ERMB_DONT_RENDER";
	string zprimeDOFBehaviour	= "ERMB_DONT_RENDER";
	string shadowGenBehaviour = "ERMB_DONT_RENDER";
	string lowDetailBehaviour	= "ERMB_RENDER";
	string lowDetailTechnique	= "Unlit_Trans_HDR";
	int    lowDetailDeferredID = 2;
>
{
	pass Pass0
#ifdef _3DSMAX_
	<
		bool ZEnable = true;
		bool ZWriteEnable = false;
		bool AlphaBlendEnable = true;
		string DestBlend = "INVSRCALPHA";
		string SrcBlend = "SRCALPHA";
		string BlendOp = "ADD";
	>
#endif
	{
#ifdef _3DSMAX_
		ZEnable = true;
		ZWriteEnable = false;
		AlphaBlendEnable = true;
		DestBlend = INVSRCALPHA;
		SrcBlend = SRCALPHA;
		BlendOp = ADD;
#endif

#if defined (_PS3_)		
		VertexShader = compile sce_vp_rsx UnlitVertexShader();
		PixelShader = compile sce_fp_rsx UnlitHDRFragmentShader();
#else		
		VertexShader = compile vs_3_0 UnlitVertexShader();
		PixelShader = compile ps_3_0 UnlitHDRFragmentShader();
#endif		
	}
}



technique _Unlit_LowDetail
<
	bool preservesGlobalState = true;
>
{
	pass Pass0
	{
#if defined (_PS3_)
		VertexShader = compile sce_vp_rsx UnlitVertexShader();
		PixelShader = compile sce_fp_rsx UnlitLowDetailFragmentShader();
#else
		VertexShader = compile vs_3_0 UnlitVertexShader();
		PixelShader = compile ps_3_0 UnlitLowDetailFragmentShader();
#endif
	}
}
